/**
  ******************************************************************************
  * @file    py32f092_hal_comp.h
  * @author  MCU Application Team
  * @brief   Header file of COMP HAL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32F092_HAL_COMP_H
#define __PY32F092_HAL_COMP_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32f092_hal_def.h"
#include "py32f092_ll_exti.h"

/** @addtogroup PY32F092_HAL_Driver
  * @{
  */


/** @addtogroup COMP
  * @{
  */

/* Exported types ------------------------------------------------------------*/
/** @defgroup COMP_Exported_Types COMP Exported Types
  * @{
  */

/**
  * @brief  COMP Init structure definition
  */
typedef struct
{

  uint32_t WindowMode;         /*!< Set window mode of a pair of comparators instances
                                    (2 consecutive instances odd and even COMP<x> and COMP<x+1>).
                                    Note: HAL COMP driver allows to set window mode from any COMP instance of the pair of COMP instances composing window mode.
                                    This parameter can be a value of @ref COMP_WindowMode */

  uint32_t Mode;               /*!< Set comparator operating mode to adjust power and speed.
                                    Note: For the characteristics of comparator power modes
                                          (propagation delay and power consumption), refer to device datasheet.
                                    This parameter can be a value of @ref COMP_PowerMode */

  uint32_t InputPlus;          /*!< Set comparator input plus (non-inverting input).
                                    This parameter can be a value of @ref COMP_InputPlus */

  uint32_t InputMinus;         /*!< Set comparator input minus (inverting input).
                                    This parameter can be a value of @ref COMP_InputMinus */
  
  uint32_t VrefSrc;            /*!< Set comparator Vref Source.
                                    This parameter can be a value of @ref COMP_VrefSrc */
  
  uint32_t VrefDiv;            /*!< Set comparator Vref Div.
                                    This parameter can be a value of @ref COMP_VrefDiv */
  
  uint32_t Hysteresis;         /*!< Set comparator hysteresis mode of the input minus.
                                    This parameter can be a value of @ref COMP_Hysteresis */

  uint32_t OutputPol;          /*!< Set comparator output polarity.
                                    This parameter can be a value of @ref COMP_OutputPolarity */

  uint32_t BlankingSrce;       /*!< Set comparator blanking source.
                                    This parameter can be a value of @ref COMP_BlankingSrce */

  uint32_t TimAndExtiOutSel;   /*!< Set whether the comparator output is filtered to TIM or EXTI. 
                                    This parameter can be a value of @ref COMP_TIM_EXTI_Out_Sel */ 
                                   
  uint32_t DigitalFilter;      /*!< Specifies the digital filter. the filter is prohibited 
                                    when the value is zero.
                                    This parameter must be a number between 0 and 0xFFFF */

  uint32_t TriggerMode;        /*!< Set the comparator output triggering External Interrupt Line (EXTI).
                                    This parameter can be a value of @ref COMP_EXTI_TriggerMode */

} COMP_InitTypeDef;

/**
  * @brief  HAL COMP state machine: HAL COMP states definition
  */
#define COMP_STATE_BITFIELD_LOCK  (0x10U)
typedef enum
{
  HAL_COMP_STATE_RESET             = 0x00U,                                             /*!< COMP not yet initialized                             */
  HAL_COMP_STATE_RESET_LOCKED      = (HAL_COMP_STATE_RESET | COMP_STATE_BITFIELD_LOCK), /*!< COMP not yet initialized and configuration is locked */
  HAL_COMP_STATE_READY             = 0x01U,                                             /*!< COMP initialized and ready for use                   */
  HAL_COMP_STATE_READY_LOCKED      = (HAL_COMP_STATE_READY | COMP_STATE_BITFIELD_LOCK), /*!< COMP initialized but configuration is locked         */
  HAL_COMP_STATE_BUSY              = 0x02U,                                             /*!< COMP is running                                      */
  HAL_COMP_STATE_BUSY_LOCKED       = (HAL_COMP_STATE_BUSY | COMP_STATE_BITFIELD_LOCK)   /*!< COMP is running and configuration is locked          */
} HAL_COMP_StateTypeDef;

/**
  * @brief  COMP Handle Structure definition
  */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
typedef struct __COMP_HandleTypeDef
#else
typedef struct
#endif
{
  COMP_TypeDef       *Instance;       /*!< Register base address    */
  COMP_InitTypeDef   Init;            /*!< COMP required parameters */
  HAL_LockTypeDef    Lock;            /*!< Locking object           */
  __IO HAL_COMP_StateTypeDef  State;  /*!< COMP communication state */
  __IO uint32_t      ErrorCode;       /*!< COMP error code */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
  void (* TriggerCallback)(struct __COMP_HandleTypeDef *hcomp);   /*!< COMP trigger callback */
  void (* MspInitCallback)(struct __COMP_HandleTypeDef *hcomp);   /*!< COMP Msp Init callback */
  void (* MspDeInitCallback)(struct __COMP_HandleTypeDef *hcomp); /*!< COMP Msp DeInit callback */
#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */
} COMP_HandleTypeDef;

#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
/**
  * @brief  HAL COMP Callback ID enumeration definition
  */
typedef enum
{
  HAL_COMP_TRIGGER_CB_ID                = 0x00U,  /*!< COMP trigger callback ID */
  HAL_COMP_MSPINIT_CB_ID                = 0x01U,  /*!< COMP Msp Init callback ID */
  HAL_COMP_MSPDEINIT_CB_ID              = 0x02U   /*!< COMP Msp DeInit callback ID */
} HAL_COMP_CallbackIDTypeDef;

/**
  * @brief  HAL COMP Callback pointer definition
  */
typedef  void (*pCOMP_CallbackTypeDef)(COMP_HandleTypeDef *hcomp); /*!< pointer to a COMP callback function */

#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */

/**
  * @}
  */

/* Exported constants --------------------------------------------------------*/
/** @defgroup COMP_Exported_Constants COMP Exported Constants
  * @{
  */

/** @defgroup COMP_Error_Code COMP Error Code
  * @{
  */
#define HAL_COMP_ERROR_NONE             (0x00UL)  /*!< No error */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
#define HAL_COMP_ERROR_INVALID_CALLBACK (0x01UL)  /*!< Invalid Callback error */
#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */
/**
  * @}
  */

/** @defgroup COMP_WindowMode COMP Window Mode
  * @{
  */
#define COMP_WINDOWMODE_DISABLE                 (0x00000000UL)         /*!< Window mode disable: Comparators instances pair COMP1 and COMP2 are independent */
#define COMP_WINDOWMODE_COMP1_INPUT_PLUS_COMMON (COMP_CSR_WINMODE | COMP_WINDOWMODE_COMP1) /*!< Window mode enable: Comparators instances pair COMP1 and COMP2 have their input plus connected together. The common input is COMP1 input plus (COMP2 input plus is no more accessible). */
#define COMP_WINDOWMODE_COMP2_INPUT_PLUS_COMMON (COMP_CSR_WINMODE | COMP_WINDOWMODE_COMP2) /*!< Window mode enable: Comparators instances pair COMP1 and COMP2 have their input plus connected together. The common input is COMP2 input plus (COMP1 input plus is no more accessible). */

/**
  * @}
  */

/** @defgroup COMP_PowerMode COMP power mode
  * @{
  */
/* Note: For the characteristics of comparator power modes                    */
/*       (propagation delay and power consumption),                           */
/*       refer to device datasheet.                                           */
#define COMP_POWERMODE_HIGHSPEED       (0x00000000UL)         /*!< High Speed */
#define COMP_POWERMODE_MEDIUMSPEED     (COMP_CSR_PWRMODE)   /*!< Medium Speed */
/**
  * @}
  */

/** @defgroup COMP_InputPlus COMP input plus (non-inverting input)
  * @{
  */ 
#define COMP_INPUT_PLUS_INTERNAL0  (0x00000000UL)                                                 /*!< Comparator input plus connected to internal analog module(OPA OUT for COMP1, DAC OUT for COMP2) */
#define COMP_INPUT_PLUS_INTERNAL1  (COMP_CSR_INPSEL_0)                                            /*!< Comparator input plus connected to internal analog module(DAC1 OUT for COMP1, OPA OUT for COMP2) */
#define COMP_INPUT_PLUS_IO0        (COMP_CSR_INPSEL_1)                                            /*!< Comparator input plus connected to IO0(pin PC15 for COMP1, pin PB14 for COMP2) */
#define COMP_INPUT_PLUS_IO1        (COMP_CSR_INPSEL_0 | COMP_CSR_INPSEL_1)                        /*!< Comparator input plus connected to IO1(pin PD0 for COMP1, pin PC3 for COMP2) */
#define COMP_INPUT_PLUS_IO2        (COMP_CSR_INPSEL_2)                                            /*!< Comparator input plus connected to IO2(pin PD1 for COMP1, pin PC4 for COMP2) */

/**
  * @}
  */

/** @defgroup COMP_InputMinus COMP input minus (inverting input)
  * @{
  */
#define COMP_INPUT_MINUS_IO0        (0x00000000UL)                                              /*!< Comparator input minus connected to IO0(pin PC5 for all) */
#define COMP_INPUT_MINUS_IO1        (                                        COMP_CSR_INMSEL_0) /*!< Comparator input minus connected to IO1(pin PA1 for all) */
#define COMP_INPUT_MINUS_DAC_VOUT   (                    COMP_CSR_INMSEL_1                    ) /*!< Comparator input minus connected to IO2(DAC VOUT for all) */ 
#define COMP_INPUT_MINUS_VREFCMP    (                    COMP_CSR_INMSEL_1 | COMP_CSR_INMSEL_0) /*!< Comparator input minus connected to IO3(VREFCMP for all) */
#define COMP_INPUT_MINUS_TS_VIN     (COMP_CSR_INMSEL_2                                        ) /*!< Comparator input minus connected to IO4(TS_VIN for all) */ 
#define COMP_INPUT_MINUS_VREFINT    (COMP_CSR_INMSEL_2                     | COMP_CSR_INMSEL_0) /*!< Comparator input minus connected to IO5(VREFINT for all) */ 
#if defined(VREFBUF)
#define COMP_INPUT_MINUS_VREFBUF    (COMP_CSR_INMSEL_2 | COMP_CSR_INMSEL_1                    ) /*!< Comparator input minus connected to IO5(VREFBUF for all) */ 
#endif
#define COMP_INPUT_MINUS_IO2        (COMP_CSR_INMSEL_2 | COMP_CSR_INMSEL_1 | COMP_CSR_INMSEL_0) /*!< Comparator input minus connected to IO1(pin PA13 for all) */

/**
  * @}
  */

/** @defgroup COMP_VrefSrc COMP Vref Source
  * @{
  */
#define COMP_VREF_SOURCE_VCC            (COMP_CCSR_SSEL) 
#if defined(VREFBUF)  
#define COMP_VREF_SOURCE_VREFBUF1P024V  (VREFBUF_CR_VREFBUF_EN)                                                                   /* Vrefbuf 1.024V */  
#define COMP_VREF_SOURCE_VREFBUF1P5V    (VREFBUF_CR_VREFBUF_EN | VREFBUF_CR_VREFBUF_OUT_SEL_0)                                    /* Vrefbuf 1.5V */  
#define COMP_VREF_SOURCE_VREFBUF2P048V  (VREFBUF_CR_VREFBUF_EN | VREFBUF_CR_VREFBUF_OUT_SEL_1)                                    /* Vrefbuf 2.048V */  
#define COMP_VREF_SOURCE_VREFBUF2P5V    (VREFBUF_CR_VREFBUF_EN | VREFBUF_CR_VREFBUF_OUT_SEL_1 | VREFBUF_CR_VREFBUF_OUT_SEL_0)     /* Vrefbuf 2.5V */  
#define COMP_VREF_SOURCE_VREFBUF0P6V    (VREFBUF_CR_VREFBUF_EN | VREFBUF_CR_VREFBUF_OUT_SEL_2)                                    /* Vrefbuf 0.6V */  
#endif
/**
  * @}
  */

/** @defgroup COMP_VrefDiv COMP Vref Div
  * @{
  */
#define COMP_VREFCMP_DIV_1_64VREFCMP    (0x00000000U)
#define COMP_VREFCMP_DIV_2_64VREFCMP    (                                                                                                    COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_3_64VREFCMP    (                                                                                COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_4_64VREFCMP    (                                                                                COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_5_64VREFCMP    (                                                            COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_6_64VREFCMP    (                                                            COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_7_64VREFCMP    (                                                            COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_8_64VREFCMP    (                                                            COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_9_64VREFCMP    (                                        COMP_CCSR_VCDIV_3                                                            )
#define COMP_VREFCMP_DIV_10_64VREFCMP   (                                        COMP_CCSR_VCDIV_3                                         | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_11_64VREFCMP   (                                        COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_12_64VREFCMP   (                                        COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_13_64VREFCMP   (                                        COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_14_64VREFCMP   (                                        COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_15_64VREFCMP   (                                        COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_16_64VREFCMP   (                                        COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_17_64VREFCMP   (                    COMP_CCSR_VCDIV_4                                                                                )
#define COMP_VREFCMP_DIV_18_64VREFCMP   (                    COMP_CCSR_VCDIV_4                                                             | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_19_64VREFCMP   (                    COMP_CCSR_VCDIV_4                                         | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_20_64VREFCMP   (                    COMP_CCSR_VCDIV_4                                         | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_21_64VREFCMP   (                    COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_22_64VREFCMP   (                    COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_23_64VREFCMP   (                    COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_24_64VREFCMP   (                    COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_25_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                                                            )
#define COMP_VREFCMP_DIV_26_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                                         | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_27_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_28_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_29_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_30_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_31_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_32_64VREFCMP   (                    COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_33_64VREFCMP   (COMP_CCSR_VCDIV_5                                                                                                    )
#define COMP_VREFCMP_DIV_34_64VREFCMP   (COMP_CCSR_VCDIV_5                                                                                 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_35_64VREFCMP   (COMP_CCSR_VCDIV_5                                                             | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_36_64VREFCMP   (COMP_CCSR_VCDIV_5                                                             | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_37_64VREFCMP   (COMP_CCSR_VCDIV_5                                         | COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_38_64VREFCMP   (COMP_CCSR_VCDIV_5                                         | COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_39_64VREFCMP   (COMP_CCSR_VCDIV_5                                         | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_40_64VREFCMP   (COMP_CCSR_VCDIV_5                                         | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_41_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3                                                            )
#define COMP_VREFCMP_DIV_42_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3                                         | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_43_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_44_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_45_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_46_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_47_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_48_64VREFCMP   (COMP_CCSR_VCDIV_5                     | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_49_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                                                                                )
#define COMP_VREFCMP_DIV_50_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                                                             | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_51_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                                         | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_52_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                                         | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_53_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_54_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_55_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_56_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4                     | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_57_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                                                            )
#define COMP_VREFCMP_DIV_58_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                                         | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_59_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_60_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3                     | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_61_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                                        )
#define COMP_VREFCMP_DIV_62_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2                     | COMP_CCSR_VCDIV_0)
#define COMP_VREFCMP_DIV_63_64VREFCMP   (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1                    )
#define COMP_VREFCMP_DIV_VREFCMP        (COMP_CCSR_VCDIV_5 | COMP_CCSR_VCDIV_4 | COMP_CCSR_VCDIV_3 | COMP_CCSR_VCDIV_2 | COMP_CCSR_VCDIV_1 | COMP_CCSR_VCDIV_0)

/**
  * @}
  */

/** @defgroup COMP_Hysteresis COMP hysteresis
  * @{
  */
#define COMP_HYSTERESIS_DISABLE           (0x00000000UL)                       /*!< No hysteresis */
#define COMP_HYSTERESIS_ENABLE            (0x00000001UL)                       /*!< Hysteresis enable */  
/**
  * @}
  */

/** @defgroup COMP_TIM_EXTI_Out_Sel COMP TIM EXTI Output Selection
  * @{
  */
#define COMP_TIM_EXTI_OUT_NOFILTER                 (0x00000000UL)                      /*!< The comparator output to TIM or EXTI is no filtered */
#define COMP_TIM_EXTI_OUT_FILTER                   (COMP_CSR_INT_OUT_SEL)              /*!< The comparator output to TIM or EXTI is filtered */
/**
  * @} 
  */  

/** @defgroup COMP_OutputPolarity COMP output Polarity
  * @{
  */
#define COMP_OUTPUTPOL_NONINVERTED     (0x00000000UL)         /*!< COMP output level is not inverted (comparator output is high when the input plus is at a higher voltage than the input minus) */
#define COMP_OUTPUTPOL_INVERTED        (COMP_CSR_POLARITY)    /*!< COMP output level is inverted     (comparator output is low  when the input plus is at a higher voltage than the input minus) */
/**
  * @}
  */

/** @defgroup COMP_BlankingSrce  COMP blanking source
  * @{
  */
#define COMP_BLANKINGSRC_NONE             (0x00000000UL)          /*!<Comparator output without blanking */
#define COMP_BLANKINGSRC_TIM1_OC4         (                                            COMP_CSR_BLANKSEL_0)   /*!< Comparator output blanking source TIM1 OC5 (specific to COMP instance: COMP1). Note: For COMPx & TIMx instances availability, please refer to datasheet */
#define COMP_BLANKINGSRC_TIM2_OC3         (                      COMP_CSR_BLANKSEL_1                      )   /*!< Comparator output blanking source TIM1 OC5 (specific to COMP instance: COMP2). Note: For COMPx & TIMx instances availability, please refer to datasheet */
#define COMP_BLANKINGSRC_TIM3_OC3         (                      COMP_CSR_BLANKSEL_1 | COMP_CSR_BLANKSEL_0)   /*!< Comparator output blanking source TIM1 OC5 (specific to COMP instance: COMP3). Note: For COMPx & TIMx instances availability, please refer to datasheet */
#define COMP_BLANKINGSRC_TIM15_OC2        (COMP_CSR_BLANKSEL_2                                            )   /*!< Comparator output blanking source TIM3 OC4 (specific to COMP instance: COMP4). Note: For COMPx & TIMx instances availability, please refer to datasheet */
#define COMP_BLANKINGSRC_PWM_OC3          (COMP_CSR_BLANKSEL_2 |                       COMP_CSR_BLANKSEL_0)   /*!< Comparator output blanking source TIM3 OC4 (specific to COMP instance: COMP4). Note: For COMPx & TIMx instances availability, please refer to datasheet */

/**
  * @}
  */

/** @defgroup COMP_OutputLevel COMP Output Level
  * @{
  */
/* Note: Comparator output level values are fixed to "0" and "1",             */
/* corresponding COMP register bit is managed by HAL function to match        */
/* with these values (independently of bit position in register).             */

/* When output polarity is not inverted, comparator output is low when
   the input plus is at a lower voltage than the input minus */
#define COMP_OUTPUT_LEVEL_LOW              (0x00000000UL)
/* When output polarity is not inverted, comparator output is high when
   the input plus is at a higher voltage than the input minus */
#define COMP_OUTPUT_LEVEL_HIGH             (0x00000001UL)
/**
  * @}
  */

/** @defgroup COMP_EXTI_TriggerMode COMP output to EXTI
  * @{
  */
#define COMP_TRIGGERMODE_NONE                 (0x00000000UL)                                            /*!< Comparator output triggering no External Interrupt Line */
#define COMP_TRIGGERMODE_IT_RISING            (COMP_EXTI_IT | COMP_EXTI_RISING)                         /*!< Comparator output triggering External Interrupt Line event with interruption, on rising edge */
#define COMP_TRIGGERMODE_IT_FALLING           (COMP_EXTI_IT | COMP_EXTI_FALLING)                        /*!< Comparator output triggering External Interrupt Line event with interruption, on falling edge */
#define COMP_TRIGGERMODE_IT_RISING_FALLING    (COMP_EXTI_IT | COMP_EXTI_RISING | COMP_EXTI_FALLING)     /*!< Comparator output triggering External Interrupt Line event with interruption, on both rising and falling edges */
#define COMP_TRIGGERMODE_EVENT_RISING         (COMP_EXTI_EVENT | COMP_EXTI_RISING)                      /*!< Comparator output triggering External Interrupt Line event only (without interruption), on rising edge */
#define COMP_TRIGGERMODE_EVENT_FALLING        (COMP_EXTI_EVENT | COMP_EXTI_FALLING)                     /*!< Comparator output triggering External Interrupt Line event only (without interruption), on falling edge */
#define COMP_TRIGGERMODE_EVENT_RISING_FALLING (COMP_EXTI_EVENT | COMP_EXTI_RISING | COMP_EXTI_FALLING)  /*!< Comparator output triggering External Interrupt Line event only (without interruption), on both rising and falling edges */
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/
/** @defgroup COMP_Exported_Macros COMP Exported Macros
  * @{
  */

/** @defgroup COMP_Handle_Management  COMP Handle Management
  * @{
  */

/** @brief  Reset COMP handle state.
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
#define __HAL_COMP_RESET_HANDLE_STATE(__HANDLE__) do{                                                  \
                                                      (__HANDLE__)->State = HAL_COMP_STATE_RESET;      \
                                                      (__HANDLE__)->MspInitCallback = NULL;            \
                                                      (__HANDLE__)->MspDeInitCallback = NULL;          \
                                                    } while(0)
#else
#define __HAL_COMP_RESET_HANDLE_STATE(__HANDLE__) ((__HANDLE__)->State = HAL_COMP_STATE_RESET)
#endif

/**
  * @brief Clear COMP error code (set it to no error code "HAL_COMP_ERROR_NONE").
  * @param __HANDLE__ COMP handle
  * @retval None
  */
#define COMP_CLEAR_ERRORCODE(__HANDLE__) ((__HANDLE__)->ErrorCode = HAL_COMP_ERROR_NONE)

/**
  * @brief  Enable the specified comparator.
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#define __HAL_COMP_ENABLE(__HANDLE__)              SET_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_EN)

/**
  * @brief  Disable the specified comparator.
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#define __HAL_COMP_DISABLE(__HANDLE__)             CLEAR_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_EN)
/**
  * @brief  Lock the specified comparator configuration.
  * @note   Using this macro induce HAL COMP handle state machine being no
  *         more in line with COMP instance state.
  *         To keep HAL COMP handle state machine updated, it is recommended
  *         to use function "HAL_COMP_Lock')".
  * @param  __HANDLE__  COMP handle
  * @retval None
  */
#define __HAL_COMP_LOCK(__HANDLE__)                SET_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_LOCK)

/**
  * @brief  Check whether the specified comparator is locked.
  * @param  __HANDLE__  COMP handle
  * @retval Value 0 if COMP instance is not locked, value 1 if COMP instance is locked
  */
#define __HAL_COMP_IS_LOCKED(__HANDLE__)           (READ_BIT((__HANDLE__)->Instance->CSR, COMP_CSR_LOCK) == COMP_CSR_LOCK)

/**
  * @}
  */

/** @defgroup COMP_Exti_Management  COMP external interrupt line management
  * @{
  */

/**
  * @brief  Enable the COMP1 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_RISING_EDGE()    LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_RISING_EDGE()   LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Enable the COMP1 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_FALLING_EDGE()   LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_FALLING_EDGE()  LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Enable the COMP1 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_RISING_FALLING_EDGE()   do { \
                                                               LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP1); \
                                                               LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP1); \
                                                             } while(0)

/**
  * @brief  Disable the COMP1 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_RISING_FALLING_EDGE()  do { \
                                                               LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP1); \
                                                               LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP1); \
                                                             } while(0)

/**
  * @brief  Enable the COMP1 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_IT()             LL_EXTI_EnableIT(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_IT()            LL_EXTI_DisableIT(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Generate a software interrupt on the COMP1 EXTI line.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_GENERATE_SWIT()         LL_EXTI_GenerateSWI(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Enable the COMP1 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_ENABLE_EVENT()          LL_EXTI_EnableEvent(COMP_EXTI_LINE_COMP1)

/**
  * @brief  Disable the COMP1 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP1_EXTI_DISABLE_EVENT()         LL_EXTI_DisableEvent(COMP_EXTI_LINE_COMP1)


/**
  * @brief  Enable the COMP2 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_RISING_EDGE()    LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line rising edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_RISING_EDGE()   LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Enable the COMP2 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_FALLING_EDGE()   LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_FALLING_EDGE()  LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Enable the COMP2 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_RISING_FALLING_EDGE()   do { \
                                                               LL_EXTI_EnableRisingTrig(COMP_EXTI_LINE_COMP2); \
                                                               LL_EXTI_EnableFallingTrig(COMP_EXTI_LINE_COMP2); \
                                                             } while(0)

/**
  * @brief  Disable the COMP2 EXTI line rising & falling edge trigger.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_RISING_FALLING_EDGE()  do { \
                                                               LL_EXTI_DisableRisingTrig(COMP_EXTI_LINE_COMP2); \
                                                               LL_EXTI_DisableFallingTrig(COMP_EXTI_LINE_COMP2); \
                                                             } while(0)

/**
  * @brief  Enable the COMP2 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_IT()             LL_EXTI_EnableIT(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line in interrupt mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_IT()            LL_EXTI_DisableIT(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Generate a software interrupt on the COMP2 EXTI line.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_GENERATE_SWIT()         LL_EXTI_GenerateSWI(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Enable the COMP2 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_ENABLE_EVENT()          LL_EXTI_EnableEvent(COMP_EXTI_LINE_COMP2)

/**
  * @brief  Disable the COMP2 EXTI line in event mode.
  * @retval None
  */
#define __HAL_COMP_COMP2_EXTI_DISABLE_EVENT()         LL_EXTI_DisableEvent(COMP_EXTI_LINE_COMP2)

/**
  * @}
  */

/**
  * @}
  */


/* Private types -------------------------------------------------------------*/
/* Private constants ---------------------------------------------------------*/
/** @defgroup COMP_Private_Constants COMP Private Constants
  * @{
  */

/** @defgroup COMP_WindowMode_Instance_Differentiator COMP window mode instance differentiator
  * @{
  */
#define COMP_WINDOWMODE_COMP1          0x00001000U       /*!< COMP window mode using common input of COMP instance: COMP1 */
#define COMP_WINDOWMODE_COMP2          0x00002000U       /*!< COMP window mode using common input of COMP instance: COMP2 */
/**
  * @}
  */

/** @defgroup COMP_ExtiLine COMP EXTI Lines
  * @{
  */
#define COMP_EXTI_LINE_COMP1           (EXTI_IMR_IM17)  /*!< EXTI line 18 connected to COMP1 output */
#define COMP_EXTI_LINE_COMP2           (EXTI_IMR_IM18)  /*!< EXTI line 19 connected to COMP2 output */

/**
  * @}
  */

/** @defgroup COMP_ExtiLine COMP EXTI Lines
  * @{
  */
#define COMP_EXTI_IT                        (0x00000001UL)  /*!< EXTI line event with interruption */
#define COMP_EXTI_EVENT                     (0x00000002UL)  /*!< EXTI line event only (without interruption) */
#define COMP_EXTI_RISING                    (0x00000010UL)  /*!< EXTI line event on rising edge */
#define COMP_EXTI_FALLING                   (0x00000020UL)  /*!< EXTI line event on falling edge */
/**
  * @}
  */

/**
  * @}
  */

/* Private macros ------------------------------------------------------------*/
/** @defgroup COMP_Private_Macros COMP Private Macros
  * @{
  */

/** @defgroup COMP_GET_EXTI_LINE COMP private macros to get EXTI line associated with comparators
  * @{
  */
/**
  * @brief  Get the specified EXTI line for a comparator instance.
  * @param  __INSTANCE__  specifies the COMP instance.
  * @retval value of @ref COMP_ExtiLine
  */

#define COMP_GET_EXTI_LINE(__INSTANCE__)    (((__INSTANCE__) == COMP1) ? COMP_EXTI_LINE_COMP1  \
                                              :COMP_EXTI_LINE_COMP2)


/**
  * @}
  */

/** @defgroup COMP_IS_COMP_Definitions COMP private macros to check input parameters
  * @{
  */
#define IS_COMP_WINDOWMODE(__WINDOWMODE__)  (((__WINDOWMODE__) == COMP_WINDOWMODE_DISABLE)                || \
                                             ((__WINDOWMODE__) == COMP_WINDOWMODE_COMP1_INPUT_PLUS_COMMON)|| \
                                             ((__WINDOWMODE__) == COMP_WINDOWMODE_COMP2_INPUT_PLUS_COMMON)  )

#define IS_COMP_POWERMODE(__POWERMODE__)    (((__POWERMODE__) == COMP_POWERMODE_HIGHSPEED)    || \
                                             ((__POWERMODE__) == COMP_POWERMODE_MEDIUMSPEED)    )

#define IS_COMP_INPUT_PLUS(__COMP_INSTANCE__, __INPUT_PLUS__) (((__INPUT_PLUS__) == COMP_INPUT_PLUS_INTERNAL0)  || \
                                                               ((__INPUT_PLUS__) == COMP_INPUT_PLUS_INTERNAL1)  || \
                                                               ((__INPUT_PLUS__) == COMP_INPUT_PLUS_IO0)  || \
                                                               ((__INPUT_PLUS__) == COMP_INPUT_PLUS_IO1)  || \
                                                               ((__INPUT_PLUS__) == COMP_INPUT_PLUS_IO2))

#if defined(VREFBUF)  
#define IS_COMP_INPUT_MINUS(__COMP_INSTANCE__, __INPUT_MINUS__) (((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO0)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO1)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_DAC_VOUT) || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_VREFCMP)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_TS_VIN)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_VREFINT) || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_VREFBUF) || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO2))
#else
#define IS_COMP_INPUT_MINUS(__COMP_INSTANCE__, __INPUT_MINUS__) (((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO0)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO1)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_DAC_VOUT) || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_VREFCMP)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_TS_VIN)  || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_VREFINT) || \
                                                                 ((__INPUT_MINUS__) == COMP_INPUT_MINUS_IO2))
#endif
#define IS_COMP_HYSTERESIS(__HYSTERESIS__)  (((__HYSTERESIS__) == COMP_HYSTERESIS_DISABLE)   || \
                                             ((__HYSTERESIS__) == COMP_HYSTERESIS_ENABLE))

#if defined(VREFBUF)  
#define IS_COMP_VREFSRC(__VREFSRC__)         (((__VREFSRC__) == COMP_VREF_SOURCE_VCC)   || \
                                              ((__VREFSRC__) == COMP_VREF_SOURCE_VREFBUF1P024V) || \
                                              ((__VREFSRC__) == COMP_VREF_SOURCE_VREFBUF1P5V)   || \
                                              ((__VREFSRC__) == COMP_VREF_SOURCE_VREFBUF2P048V) || \
                                              ((__VREFSRC__) == COMP_VREF_SOURCE_VREFBUF2P5V)   || \
                                              ((__VREFSRC__) == COMP_VREF_SOURCE_VREFBUF0P6V)) 
#else
#define IS_COMP_VREFSRC(__VREFSRC__)         (((__VREFSRC__) == COMP_VREF_SOURCE_VCC)) 
#endif                                                

#define IS_COMP_VREFDIV(__VREFCMP__)         (((__VREFCMP__) == COMP_VREFCMP_DIV_1_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_2_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_3_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_4_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_5_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_6_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_7_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_8_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_9_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_10_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_11_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_12_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_13_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_14_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_15_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_16_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_17_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_18_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_19_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_20_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_21_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_22_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_23_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_24_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_25_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_26_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_27_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_28_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_29_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_30_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_31_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_32_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_33_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_34_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_35_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_36_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_37_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_38_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_39_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_40_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_41_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_42_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_43_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_44_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_45_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_46_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_47_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_48_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_49_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_50_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_51_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_52_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_53_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_54_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_55_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_56_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_57_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_58_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_59_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_60_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_61_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_62_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_63_64VREFCMP)   || \
                                              ((__VREFCMP__) == COMP_VREFCMP_DIV_VREFCMP))

#define IS_COMP_OUTPUTPOL(__POL__)          (((__POL__) == COMP_OUTPUTPOL_NONINVERTED) || \
                                             ((__POL__) == COMP_OUTPUTPOL_INVERTED))


#define IS_COMP_BLANKINGSRCE(__BLANKINGSRCE__)          (((__BLANKINGSRCE__) == COMP_BLANKINGSRC_NONE) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM1_OC4) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM2_OC3) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM3_OC3) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_TIM15_OC2) || \
                                                         ((__BLANKINGSRCE__) == COMP_BLANKINGSRC_PWM_OC3))

#define IS_COMP_TIMANDEXTIOUTSEL(__TIMANDEXTIOUTSEL__)          (((__TIMANDEXTIOUTSEL__) == COMP_TIM_EXTI_OUT_NOFILTER) || \
                                                                 ((__TIMANDEXTIOUTSEL__) == COMP_TIM_EXTI_OUT_FILTER))

#define IS_COMP_TRIGGERMODE(__MODE__)       (((__MODE__) == COMP_TRIGGERMODE_NONE)                 || \
                                             ((__MODE__) == COMP_TRIGGERMODE_IT_RISING)            || \
                                             ((__MODE__) == COMP_TRIGGERMODE_IT_FALLING)           || \
                                             ((__MODE__) == COMP_TRIGGERMODE_IT_RISING_FALLING)    || \
                                             ((__MODE__) == COMP_TRIGGERMODE_EVENT_RISING)         || \
                                             ((__MODE__) == COMP_TRIGGERMODE_EVENT_FALLING)        || \
                                             ((__MODE__) == COMP_TRIGGERMODE_EVENT_RISING_FALLING))

#define IS_COMP_OUTPUT_LEVEL(__OUTPUT_LEVEL__) (((__OUTPUT_LEVEL__) == COMP_OUTPUT_LEVEL_LOW)     || \
                                                ((__OUTPUT_LEVEL__) == COMP_OUTPUT_LEVEL_HIGH))

/**
  * @}
  */

/**
  * @}
  */


/* Exported functions --------------------------------------------------------*/
/** @addtogroup COMP_Exported_Functions
  * @{
  */

/** @addtogroup COMP_Exported_Functions_Group1
  * @{
  */

/* Initialization and de-initialization functions  **********************************/
HAL_StatusTypeDef HAL_COMP_Init(COMP_HandleTypeDef *hcomp);
HAL_StatusTypeDef HAL_COMP_DeInit(COMP_HandleTypeDef *hcomp);
void              HAL_COMP_MspInit(COMP_HandleTypeDef *hcomp);
void              HAL_COMP_MspDeInit(COMP_HandleTypeDef *hcomp);

#if (USE_HAL_COMP_REGISTER_CALLBACKS == 1)
/* Callbacks Register/UnRegister functions  ***********************************/
HAL_StatusTypeDef HAL_COMP_RegisterCallback(COMP_HandleTypeDef *hcomp, HAL_COMP_CallbackIDTypeDef CallbackID,
    pCOMP_CallbackTypeDef pCallback);
HAL_StatusTypeDef HAL_COMP_UnRegisterCallback(COMP_HandleTypeDef *hcomp, HAL_COMP_CallbackIDTypeDef CallbackID);
#endif /* USE_HAL_COMP_REGISTER_CALLBACKS */
/**
  * @}
  */

/* IO operation functions  *****************************************************/
/** @addtogroup COMP_Exported_Functions_Group2
  * @{
  */
HAL_StatusTypeDef HAL_COMP_Start(COMP_HandleTypeDef *hcomp);
HAL_StatusTypeDef HAL_COMP_Stop(COMP_HandleTypeDef *hcomp);
void              HAL_COMP_IRQHandler(COMP_HandleTypeDef *hcomp);
/**
  * @}
  */

/* Peripheral Control functions  ************************************************/
/** @addtogroup COMP_Exported_Functions_Group3
  * @{
  */
HAL_StatusTypeDef HAL_COMP_Lock(COMP_HandleTypeDef *hcomp);
uint32_t          HAL_COMP_GetOutputLevel(COMP_HandleTypeDef *hcomp);
/* Callback in interrupt mode */
void              HAL_COMP_TriggerCallback(COMP_HandleTypeDef *hcomp);
/**
  * @}
  */

/* Peripheral State functions  **************************************************/
/** @addtogroup COMP_Exported_Functions_Group4
  * @{
  */
HAL_COMP_StateTypeDef HAL_COMP_GetState(COMP_HandleTypeDef *hcomp);
uint32_t              HAL_COMP_GetError(COMP_HandleTypeDef *hcomp);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* PY32F092_HAL_COMP_H */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
